#version 300 es
precision highp float;
in vec3 vColor;
in vec2 vUV;
in vec3 vNormal;
in vec3 vPosition;
in vec3 vWPosition;
in vec3 vTangent;

uniform float time;
uniform sampler2D BaseColor;
uniform sampler2D MetallicRoughness;
uniform sampler2D Normal;
uniform sampler2D Occlusion;
uniform sampler2D Emissive;

out vec4 fragColor;

//variables carried through the process for rendering our output.
struct Process{
    vec3 lightDir;
    vec3 color;
    vec4 baseColor;
    vec4 emissiveColor;
    float metallic;
    float roughness;
    vec3 normal;
};

//const variable to use as defines
const vec3 viewDir = vec3(0.0, 0.0, 1.0);
const vec3 ambiCol = vec3(0.01, 0.025, 0.05);

vec2 rotate2(vec2 inCoords, float r)
{
    float s = sin(r);
    float c = cos(r);
    return mat2(c, -s,
                s,  c) * inCoords;
}

void SetBaseColor(inout Process p)
{
    p.baseColor = texture(BaseColor, vUV);
    if (any(isnan(p.baseColor))) {
        p.baseColor = vec4(1., 0., 1., 1.);
    }
    p.color = p.baseColor.rgb;
}

void SetMetallicRoughness(inout Process p)
{
    vec4 texMR = texture(MetallicRoughness, vUV);
    if (any(isnan(texMR.rgb))) {
        texMR = vec4(0., 1., 0., 0.);
    }
    p.metallic = texMR.r;
    p.roughness = texMR.g;
}

//something seems strange about the way GLTF is getting this
//could be a bad way that blender is exporting it.
//for now I'll do a clear coat.
//with a very fake sampling for reflectiveness
void SetMetallic(inout Process p)
{
    float f = dot(p.normal, viewDir);
    vec4 refColor = texture(BaseColor, vUV.yx + f);
    if (any(isnan(refColor))) {
        refColor = vec4(1., 0., 1., 1.);
    }
    p.color += refColor.rgb * ambiCol * p.metallic;
}

void SetNormal(inout Process p)
{
    vec3 texNormal = texture(Normal, vUV).rgb;

    if (any(isnan(texNormal))) {
        texNormal = vec3(0.5, 0.5, 1.0);
    }

    // Tangent-space normal
    vec3 tsNormal = texNormal * 2.0 - 1.0;

    vec3 n = normalize(vNormal);
    vec3 t = normalize(vTangent.xyz);
    vec3 b = normalize(cross(n, t));

    p.normal = normalize(mat3(t, b, n) * tsNormal);
}

void SetLight(inout Process p)
{
    float nDotl = dot(p.normal, p.lightDir) * 0.5 + 0.5;
    nDotl = clamp(nDotl, 0.0, 1.0);

    //set base color by directional light
    p.color = p.color * nDotl;
    
    //add ambient light
    p.color += (1.0 - nDotl) * ambiCol;
}

void SetSpecular(inout Process p)
{
    float specStr = pow(1.0 - p.roughness, 4.0); // specular strength based on roughness
    float specPower = (1.0 - p.roughness) * 8.0; // specular power
    specPower = pow(2.0, specPower); 
    
    vec3 reflect = reflect(p.lightDir, p.normal);
    float spec = dot(reflect, normalize(viewDir));
    spec = pow(spec, specPower);
    spec = clamp(spec, 0., 1.);
    spec = smoothstep(0., .001, spec);
    spec *= specStr;
    p.color += spec;
}

void SetEmission(inout Process p)
{
    vec4 texEmit = texture(Emissive, vUV);
    p.color += mix(p.color, texEmit.rgb, texEmit.a);
}

void main() 
{
    Process p;
    p.lightDir = normalize(vec3(0.0, -.5, -1.0) - vWPosition);
    SetBaseColor(p);
    SetMetallicRoughness(p);
    SetNormal(p);
    SetLight(p);
    SetMetallic(p);
    SetSpecular(p);
    SetEmission(p);

    //commit color
    fragColor = vec4(p.color, p.baseColor.a);
}
